<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Linkjuice
 * @subpackage Seolaxy_Linkjuice/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Seolaxy_Linkjuice
 * @subpackage Seolaxy_Linkjuice/public
 * @author     SEOLAXY
 */
class Seolaxy_Linkjuice_Public
{

  /**
   * The ID of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $plugin_name    The ID of this plugin.
   */
  private $plugin_name;

  /**
   * The version of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $version    The current version of this plugin.
   */
  private $version;

  /**
   * Unique ID for this class.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $id    The ID of this class.
   */
  private $id;

  private $seolaxy_linkjuice_options;

  private $seolaxy_linkjuice_options_default;

  private $selected_linkjuice_adjustment;

  private $selected_options;

  private $seo_conflicts_setup = false;

  /* private $custom_linkjuice_options;

  private $product_linkjuice_options;

  private $category_linkjuice_options; */

  /**
   * Initialize the class and set its properties.
   *
   * @since    1.0.0
   * @param      string $plugin_name       The name of the plugin.
   * @param      string $version    The version of this plugin.
   */
  public function __construct($plugin_name, $version)
  {
    $this->plugin_name = $plugin_name;
    $this->version = $version;
    $this->id = $this->plugin_name . '-public';

    $this->seolaxy_linkjuice_options_default = seolaxy_linkjuice_default_options();
    $this->seolaxy_linkjuice_options = get_option('seolaxy_linkjuice_options');

    $this->selected_linkjuice_adjustment = $this->seolaxy_linkjuice_options['linkjuice_adjustment'] ?? $this->seolaxy_linkjuice_options_default['linkjuice_adjustment'];
    if ($this->selected_linkjuice_adjustment == "custom") {
      $this->selected_options = $this->set_custom_linkjuice_options();
    } elseif ($this->selected_linkjuice_adjustment == "product") {
      $this->selected_options =
        $this->set_product_linkjuice_options();
    } elseif ($this->selected_linkjuice_adjustment == "category") {
      $this->selected_options =
        $this->set_category_linkjuice_options();
    }
    /* echo $this->selected_linkjuice_adjustment;
    echo "<pre>";
    print_r($this->selected_options);
    echo "</pre>";
    die(); */
    if ($this->selected_options) {
      add_action('init', array($this, 'remove_default_robot_meta'));
      // Setup conflict resolution as early as possible
      add_action('plugins_loaded', array($this, 'setup_seo_conflicts_resolution'), 999);
      // Also setup on wp hook as backup
      add_action('wp', array($this, 'setup_seo_conflicts_resolution'), 1);
      // Setup on template_redirect for additional safety
      add_action('template_redirect', array($this, 'setup_seo_conflicts_resolution'), 1);
    }
  }

  public function remove_default_robot_meta()
  {
    remove_filter('wp_robots', 'wp_robots_max_image_preview_large');
  }



  /**
   * Setup SEO plugin conflict resolution
   * 
   * Detects and handles conflicts with other SEO plugins like Rank Math and Yoast SEO
   * by ensuring our settings take priority.
   */
  public function setup_seo_conflicts_resolution()
  {
    // Prevent multiple executions
    if ($this->seo_conflicts_setup) {
      return;
    }
    $this->seo_conflicts_setup = true;

    $has_seo_conflicts = false;

    // Check if Rank Math is active
    if ($this->is_rank_math_active()) {
      // Remove Rank Math's actions early
      add_action('wp_head', array($this, 'remove_rank_math_seo_output'), 1);
      $has_seo_conflicts = true;
    }

    // Check if Yoast SEO is active
    if ($this->is_yoast_seo_active()) {
      // Remove Yoast SEO's actions early
      add_action('wp_head', array($this, 'remove_yoast_seo_output'), 1);
      $has_seo_conflicts = true;
    }

    if ($has_seo_conflicts) {
      // Add our output with higher priority (after removing other plugins)
      add_action('wp_head', array($this, 'output_robots_meta'), 5);
      add_action('wp_head', array($this, 'output_canonical_link'), 5);

      // Also override their filters with highest priority
      add_filter('wpseo_canonical', array($this, 'override_yoast_canonical'), 999);
      add_filter('wpseo_robots', array($this, 'override_yoast_robots'), 999);
      add_filter('rank_math/frontend/canonical', array($this, 'override_rankmath_canonical'), 999);
      add_filter('rank_math/frontend/robots', array($this, 'override_rankmath_robots'), 999);
    } else {
      // No conflicts detected, use standard hooks
      add_action('wp_head', array($this, 'canonical'));
      add_action('wp_robots', array($this, 'robots'));
    }
  }

  /**
   * Check if Rank Math SEO plugin is active
   * 
   * @return bool True if Rank Math is active
   */
  private function is_rank_math_active()
  {
    // By now (wp hook), all plugins should be loaded, so these checks should work
    return defined('RANK_MATH_VERSION') ||
      class_exists('RankMath') ||
      function_exists('rank_math') ||
      class_exists('\\RankMath\\Frontend\\Head');
  }

  /**
   * Check if Yoast SEO plugin is active
   * 
   * @return bool True if Yoast SEO is active
   */
  private function is_yoast_seo_active()
  {
    // Multiple checks to ensure Yoast SEO is properly detected
    return defined('WPSEO_VERSION') ||
      class_exists('WPSEO_Frontend') ||
      class_exists('WPSEO_Options') ||
      function_exists('wpseo_init') ||
      class_exists('Yoast\\WP\\SEO\\Main');
  }

  /**
   * Remove Rank Math's SEO output from their custom hooks
   * 
   * This prevents Rank Math from outputting robots and canonical tags
   * so our plugin can take over with its own settings.
   */
  public function remove_rank_math_seo_output()
  {
    // Remove from rank_math/head action using object reference
    if (function_exists('rank_math') && isset(rank_math()->head)) {
      remove_action('rank_math/head', array(rank_math()->head, 'robots'), 10);
      remove_action('rank_math/head', array(rank_math()->head, 'canonical'), 20);
    }

    // Remove all actions from rank_math/head hook for robots and canonical priorities
    remove_all_actions('rank_math/head', 10);
    remove_all_actions('rank_math/head', 20);

    // Prevent rank_math/head hook execution
    add_filter('rank_math/head', '__return_false', 1);
    add_action('rank_math/head', array($this, 'prevent_rank_math_output'), 1);
  }

  /**
   * Prevent Rank Math from outputting anything by stopping action execution
   */
  public function prevent_rank_math_output()
  {
    // Remove all subsequent actions on this hook
    remove_all_actions('rank_math/head');
  }

  /**
   * Remove Yoast SEO's output from wp_head
   * 
   * This prevents Yoast SEO from outputting robots and canonical tags
   * so our plugin can take over with its own settings.
   */
  public function remove_yoast_seo_output()
  {
    // Remove Yoast SEO's main wp_head action for older versions
    if (class_exists('WPSEO_Frontend')) {
      $yoast_frontend = WPSEO_Frontend::get_instance();
      if ($yoast_frontend) {
        // Remove canonical output
        remove_action('wp_head', array($yoast_frontend, 'canonical'), 20);
        // Remove robots output  
        remove_action('wp_head', array($yoast_frontend, 'robots'), 10);
        // Remove the main head action that outputs multiple tags
        remove_action('wp_head', array($yoast_frontend, 'head'), 1);
        // Remove additional Yoast actions
        remove_action('wp_head', array($yoast_frontend, 'debug_mark'), 2);
        remove_action('wp_head', array($yoast_frontend, 'head'), 1);
      }
    }

    // For newer Yoast versions (14+), remove from the new structure
    if (class_exists('Yoast\\WP\\SEO\\Integrations\\Front_End_Integration')) {
      remove_action('wp_head', 'Yoast\\WP\\SEO\\Integrations\\Front_End_Integration::present_head', 1);
    }

    // Remove Yoast's presenters that handle canonical and robots
    if (class_exists('Yoast\\WP\\SEO\\Presenters\\Meta_Name_Presenter')) {
      remove_action('wpseo_head', 'Yoast\\WP\\SEO\\Presenters\\Meta_Name_Presenter', 10);
    }

    if (class_exists('Yoast\\WP\\SEO\\Presenters\\Canonical_Presenter')) {
      remove_action('wpseo_head', 'Yoast\\WP\\SEO\\Presenters\\Canonical_Presenter', 10);
    }

    // Completely disable Yoast's head output by removing all their hooks
    remove_all_actions('wpseo_head');

    // Prevent wpseo_head hook execution entirely
    add_filter('wpseo_head', '__return_false', 1);

    // Also disable through Yoast's own filters
    add_filter('wpseo_canonical', '__return_false', 1);
    add_filter('wpseo_robots', '__return_false', 1);

    // Disable Yoast's frontend entirely for canonical and robots
    add_filter('wpseo_frontend_presenters', array($this, 'remove_yoast_presenters'), 999);
  }



  /**
   * Add robots directives to wp_robots filter
   * 
   * @param array $robots Current robots array
   * @return array Modified robots array
   */
  public function robots($robots)
  {
    $robots_from_settings = Robots::getRobotsByLinkjuice($this->selected_options);
    $values = explode(',', $robots_from_settings->getContent());

    $validValues = ['index', 'noindex', 'follow', 'nofollow'];

    foreach ($validValues as $value) {
      if (in_array($value, $values)) {
        $robots[$value] = true;
      }
    }
    return $robots;
  }

  /**
   * Output robots meta tag directly to wp_head
   * 
   * Used when wp_robots filter is not available (e.g., when Rank Math removes it)
   */
  public function output_robots_meta()
  {
    $robots_from_settings = Robots::getRobotsByLinkjuice($this->selected_options);
    $robots_content = trim($robots_from_settings->getContent());

    if (!empty($robots_content)) {
      echo '<meta name="robots" content="' . esc_attr($robots_content) . '" data-generator="seolaxy-linkjuice" />' . "\n";
    }
  }

  /**
   * Output canonical link tag to wp_head
   */
  public function canonical()
  {
    $this->output_canonical_link();
  }

  /**
   * Output canonical link tag directly
   * 
   * Separated into its own method for better reusability
   */
  public function output_canonical_link()
  {
    $canonical = Canonical::getCanonicalByLinkjuice($this->selected_options);
    $canonical_href = $canonical->getHref();
    $canonical_rel = $canonical->getRel();

    if (!empty($canonical_href)) {
      echo '<link rel="' . esc_attr($canonical_rel) . '" href="' . esc_url($canonical_href) . '" data-generator="seolaxy-linkjuice" />' . "\n";
    }
  }

  /**
   * Override Yoast SEO's canonical URL filter
   * 
   * @param string $canonical The canonical URL from Yoast
   * @return string Our canonical URL
   */
  public function override_yoast_canonical($canonical)
  {
    $our_canonical = Canonical::getCanonicalByLinkjuice($this->selected_options);
    $our_canonical_href = $our_canonical->getHref();

    return !empty($our_canonical_href) ? $our_canonical_href : $canonical;
  }

  /**
   * Override Yoast SEO's robots meta filter
   * 
   * @param string $robots The robots content from Yoast
   * @return string Our robots content
   */
  public function override_yoast_robots($robots)
  {
    $our_robots = Robots::getRobotsByLinkjuice($this->selected_options);
    $our_robots_content = trim($our_robots->getContent());

    return !empty($our_robots_content) ? $our_robots_content : $robots;
  }

  /**
   * Override RankMath's canonical URL filter
   * 
   * @param string $canonical The canonical URL from RankMath
   * @return string Our canonical URL
   */
  public function override_rankmath_canonical($canonical)
  {
    $our_canonical = Canonical::getCanonicalByLinkjuice($this->selected_options);
    $our_canonical_href = $our_canonical->getHref();

    return !empty($our_canonical_href) ? $our_canonical_href : $canonical;
  }

  /**
   * Override RankMath's robots meta filter
   * 
   * @param array $robots The robots array from RankMath
   * @return array Our robots array
   */
  public function override_rankmath_robots($robots)
  {
    $our_robots = Robots::getRobotsByLinkjuice($this->selected_options);
    $values = explode(',', $our_robots->getContent());

    $our_robots_array = array();
    $validValues = ['index', 'noindex', 'follow', 'nofollow'];

    foreach ($validValues as $value) {
      if (in_array($value, $values)) {
        $our_robots_array[$value] = true;
      }
    }

    return !empty($our_robots_array) ? $our_robots_array : $robots;
  }

  /**
   * Remove Yoast presenters that handle canonical and robots meta
   * 
   * @param array $presenters Array of Yoast presenters
   * @return array Filtered presenters array
   */
  public function remove_yoast_presenters($presenters)
  {
    if (!is_array($presenters)) {
      return $presenters;
    }

    // Remove canonical and robots presenters
    $presenters = array_filter($presenters, function ($presenter) {
      $presenter_class = is_object($presenter) ? get_class($presenter) : $presenter;

      // Remove canonical and robots related presenters
      return !in_array($presenter_class, [
        'Yoast\\WP\\SEO\\Presenters\\Canonical_Presenter',
        'Yoast\\WP\\SEO\\Presenters\\Meta_Name_Presenter',
        'Yoast\\WP\\SEO\\Presenters\\Robots_Presenter'
      ]);
    });

    return $presenters;
  }

  private function set_custom_linkjuice_options()
  {
    return (object)[
      "ProductCategoryPage" => $this->seolaxy_linkjuice_options['ProductCategoryPage'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryPage'],
      "ProductCategoryCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryCanonical'],
      "ProductCategoryPaging" => $this->seolaxy_linkjuice_options['ProductCategoryPaging'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryPaging'],
      "ProductCategoryPagingCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryPagingCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryPagingCanonical'],
      "ProductSubactegoryPage" => $this->seolaxy_linkjuice_options['ProductSubactegoryPage'] ?? $this->seolaxy_linkjuice_options_default['ProductSubactegoryPage'],
      "ProductSubcategoryCanonical" => $this->seolaxy_linkjuice_options['ProductSubcategoryCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductSubcategoryCanonical'],
      "ProductSubactegoryPaging" => $this->seolaxy_linkjuice_options['ProductSubactegoryPaging'] ?? $this->seolaxy_linkjuice_options_default['ProductSubactegoryPaging'],
      "ProductSubcategoryPagingCanonical" => $this->seolaxy_linkjuice_options['ProductSubcategoryPagingCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductSubcategoryPagingCanonical'],
      "ProductCategoryAndSubCategorySingleFilterPage" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySingleFilterPage'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySingleFilterPage'],
      "ProductCategoryAndSubCategorySingleFilterCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySingleFilterCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySingleFilterCanonical'],
      "ProductCategoryAndSubCategorySingleFilterPaging" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySingleFilterPaging'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySingleFilterPaging'],
      "ProductCategoryAndSubCategorySingleFilterPagingCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySingleFilterPagingCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySingleFilterPagingCanonical'],
      "ProductCategoryAndSubCategoryMultipleFilterPage" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategoryMultipleFilterPage'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategoryMultipleFilterPage'],
      "ProductCategoryAndSubCategoryMultipleFilterPageCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategoryMultipleFilterPageCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategoryMultipleFilterPageCanonical'],
      "ProductCategoryAndSubCategoryMultipleFilterPaging" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategoryMultipleFilterPaging'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategoryMultipleFilterPaging'],
      "ProductCategoryAndSubCategoryMultipleFilterPagingCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategoryMultipleFilterPagingCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategoryMultipleFilterPagingCanonical'],
      "ProductCategoryAndSubCategorySort" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySort'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySort'],
      "ProductCategoryAndSubCategorySortCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySortCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySortCanonical'],
      "ProductCategoryAndSubCategorySortPaging" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySortPaging'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySortPaging'],
      "ProductCategoryAndSubCategorySortPagingCanonical" => $this->seolaxy_linkjuice_options['ProductCategoryAndSubCategorySortPagingCanonical'] ?? $this->seolaxy_linkjuice_options_default['ProductCategoryAndSubCategorySortPagingCanonical'],
      "Search" => $this->seolaxy_linkjuice_options['Search'] ?? $this->seolaxy_linkjuice_options_default['Search'],
      "SearchCanonical" => $this->seolaxy_linkjuice_options['SearchCanonical'] ?? $this->seolaxy_linkjuice_options_default['SearchCanonical']
    ];
  }

  private function set_product_linkjuice_options()
  {
    return (object)[
      "ProductCategoryPage" => 'index,follow',
      "ProductCategoryCanonical" => 'self',
      "ProductCategoryPaging" => 'noindex,follow',
      "ProductCategoryPagingCanonical" => 'self',
      "ProductSubactegoryPage" => 'index,follow',
      "ProductSubcategoryCanonical" => 'self',
      "ProductSubactegoryPaging" => 'noindex,follow',
      "ProductSubcategoryPagingCanonical" => 'self',
      "ProductCategoryAndSubCategorySingleFilterPage" => 'index,follow',
      "ProductCategoryAndSubCategorySingleFilterCanonical" => 'self',
      "ProductCategoryAndSubCategorySingleFilterPaging" => 'noindex,follow',
      "ProductCategoryAndSubCategorySingleFilterPagingCanonical" => "self",
      "ProductCategoryAndSubCategoryMultipleFilterPage" => 'noindex,nofollow',
      "ProductCategoryAndSubCategoryMultipleFilterPageCanonical" => 'root',
      "ProductCategoryAndSubCategoryMultipleFilterPaging" => 'noindex,nofollow',
      "ProductCategoryAndSubCategoryMultipleFilterPagingCanonical" => 'root',
      "ProductCategoryAndSubCategorySort" => "noindex,nofollow",
      "ProductCategoryAndSubCategorySortCanonical" => "root",
      "ProductCategoryAndSubCategorySortPaging" => "noindex,nofollow",
      "ProductCategoryAndSubCategorySortPagingCanonical" => "root",
      "Search" => 'noindex,nofollow',
      "SearchCanonical" => 'self'
    ];
  }

  private function set_category_linkjuice_options()
  {
    return (object)[
      "ProductCategoryPage" => 'index,follow',
      "ProductCategoryCanonical" => 'self',
      "ProductCategoryPaging" => 'noindex,follow',
      "ProductCategoryPagingCanonical" => 'root',
      "ProductSubactegoryPage" => 'index,follow',
      "ProductSubcategoryCanonical" => 'self',
      "ProductSubactegoryPaging" => 'noindex,nofollow',
      "ProductSubcategoryPagingCanonical" => 'root',
      "ProductCategoryAndSubCategorySingleFilterPage" => 'index,follow',
      "ProductCategoryAndSubCategorySingleFilterCanonical" => 'self',
      "ProductCategoryAndSubCategorySingleFilterPaging" => 'noindex,nofollow',
      "ProductCategoryAndSubCategorySingleFilterPagingCanonical" => "root",
      "ProductCategoryAndSubCategoryMultipleFilterPage" => 'noindex,nofollow',
      "ProductCategoryAndSubCategoryMultipleFilterPageCanonical" => 'root',
      "ProductCategoryAndSubCategoryMultipleFilterPaging" => 'noindex,nofollow',
      "ProductCategoryAndSubCategoryMultipleFilterPagingCanonical" => 'root',
      "ProductCategoryAndSubCategorySort" => "noindex,nofollow",
      "ProductCategoryAndSubCategorySortCanonical" => "root",
      "ProductCategoryAndSubCategorySortPaging" => "noindex,nofollow",
      "ProductCategoryAndSubCategorySortPagingCanonical" => "root",
      "Search" => 'noindex,nofollow',
      "SearchCanonical" => 'home'
    ];
  }

  /**
   * Register the JavaScript and stylesheets for the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function enqueue_public_resources()
  {
    /**
     * This function is provided for demonstration purposes only.
     *
     * An instance of this class should be passed to the run() function
     * defined in Seolaxy_Linkjuice_Loader as all of the hooks are defined
     * in that particular class.
     *
     * The Seolaxy_Linkjuice_Loader will then create the relationship
     * between the defined hooks and the functions defined in this
     * class.
     */
    wp_enqueue_style($this->id, SEOLAXY_LINKJUICE_URL . 'public/css/seolaxy-linkjuice-public.css', array(), $this->version, 'all');
    wp_enqueue_script($this->id, SEOLAXY_LINKJUICE_URL . 'public/js/seolaxy-linkjuice-public.js', array('jquery'), $this->version, false);
  }
}
require_once SEOLAXY_LINKJUICE_PATH . 'public/Robots/Robots.php';
require_once SEOLAXY_LINKJUICE_PATH . 'public/Canonical/Canonical.php';
