<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Linkjuice
 * @subpackage Seolaxy_Linkjuice/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Seolaxy_Linkjuice
 * @subpackage Seolaxy_Linkjuice/includes
 * @author     SEOLAXY
 */
class Seolaxy_Linkjuice
{

  /**
   * The loader that's responsible for maintaining and registering all hooks that power
   * the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      Seolaxy_Linkjuice_Loader    $loader    Maintains and registers all hooks for the plugin.
   */
  protected $loader;

  /**
   * The unique identifier of this plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $plugin_name    The string used to uniquely identify this plugin.
   */
  protected $plugin_name;

  /**
   * The current version of the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $version    The current version of the plugin.
   */
  protected $version;

  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct()
  {
    $this->version = SEOLAXY_LINKJUICE_VERSION;
    $this->plugin_name = 'seolaxy-linkjuice';

    $this->load_dependencies();
    $this->set_locale();
    $this->define_include_hooks();
    $this->define_admin_hooks();
    $this->define_public_hooks();
  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Include the following files that make up the plugin:
   *
   * - Seolaxy_Linkjuice_Loader. Orchestrates the hooks of the plugin.
   * - Seolaxy_Linkjuice_i18n. Defines internationalization functionality.
   * - Seolaxy_Linkjuice_Admin. Defines all hooks for the admin area.
   * - Seolaxy_Linkjuice_Public. Defines all hooks for the public side of the site.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function load_dependencies()
  {

    /**
     * Plugin Core Functions.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'includes/functions.php';

    /**
     * The class responsible for orchestrating the actions and filters of the
     * core plugin.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'includes/class-seolaxy-linkjuice-loader.php';

    /**
     * The class responsible for defining internationalization functionality
     * of the plugin.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'includes/class-seolaxy-linkjuice-i18n.php';

    /**
     * The class responsible for defining all actions that occur in both admin and public-facing areas.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'includes/class-seolaxy-linkjuice-include.php';

    /**
     * The class responsible for defining all actions that occur in the admin area.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'admin/class-seolaxy-linkjuice-admin.php';

    /**
     * The class responsible for defining all actions that occur in the public-facing
     * side of the site.
     */
    require_once SEOLAXY_LINKJUICE_PATH . 'public/class-seolaxy-linkjuice-public.php';

    $this->loader = new Seolaxy_Linkjuice_Loader();
  }

  /**
   * Define the locale for this plugin for internationalization.
   *
   * Uses the Seolaxy_Linkjuice_i18n class in order to set the domain and to register the hook
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function set_locale()
  {

    $plugin_i18n = new Seolaxy_Linkjuice_i18n();

    $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
  }

  /**
   * Register all of the hooks related to both admin and public-facing areas functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_include_hooks()
  {

    $plugin_admin = new Seolaxy_Linkjuice_Include($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('init', $plugin_admin, 'init_something');
  }

  /**
   * Register all of the hooks related to the admin area functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_admin_hooks()
  {

    $plugin_admin = new Seolaxy_Linkjuice_Admin($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('admin_menu', $plugin_admin, 'add_admin_menu');
    $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_resources');

    /*Register Settings*/
    $this->loader->add_action('rest_api_init', $plugin_admin, 'register_settings');
    $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');

    $this->loader->add_action('rest_api_init', $this, 'register_rest_routes');
  }

  /**
   * Register all of the hooks related to the public-facing functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_public_hooks()
  {

    $plugin_public = new Seolaxy_Linkjuice_Public($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_public_resources');
  }

  /**
   * Run the loader to execute all of the hooks with WordPress.
   *
   * @since    1.0.0
   */
  public function run()
  {
    $this->loader->run();
  }

  /**
   * The name of the plugin used to uniquely identify it within the context of
   * WordPress and to define internationalization functionality.
   *
   * @since     1.0.0
   * @return    string    The name of the plugin.
   */
  public function get_plugin_name()
  {
    return $this->plugin_name;
  }

  /**
   * The reference to the class that orchestrates the hooks with the plugin.
   *
   * @since     1.0.0
   * @return    Seolaxy_Linkjuice_Loader    Orchestrates the hooks of the plugin.
   */
  public function get_loader()
  {
    return $this->loader;
  }

  /**
   * Retrieve the version number of the plugin.
   *
   * @since     1.0.0
   * @return    string    The version number of the plugin.
   */
  public function get_version()
  {
    return $this->version;
  }

  public function register_rest_routes()
  {
    // Add license verification endpoint - updated pattern to handle URL-encoded keys
    register_rest_route($this->plugin_name . '/v1', '/verify-license/(?P<key>[^/]+)', [
      'methods' => 'GET',
      'callback' => [$this, 'verify_license'],
      'permission_callback' => function () {
        return current_user_can('manage_options');
      },
      'args' => [
        'key' => [
          'required' => true,
          'validate_callback' => function ($param) {
            return is_string($param);
          },
          'sanitize_callback' => function ($param) {
            // URL decode the license key for processing
            return urldecode($param);
          }
        ]
      ]
    ]);
  }

  public function verify_license($request)
  {
    $license_key = $request->get_param('key');
    $domain = parse_url(get_site_url(), PHP_URL_HOST);

    // URL encode the license key for the external API call
    $encoded_license_key = urlencode($license_key);

    $response = wp_remote_get(
      "https://api.bigburg.bosstech.info/plugin/verify/{$encoded_license_key}/{$domain}",
      array(
        'timeout' => 15,
        'headers' => array(
          'Accept' => 'application/json'
        )
      )
    );

    if (is_wp_error($response)) {
      return new WP_Error('license_verification_failed', $response->get_error_message());
    }

    $body = wp_remote_retrieve_body($response);

    if ($body === 'false') {
      return new WP_REST_Response([
        'active' => false,
        'error' => 'Invalid license key'
      ], 200);
    }

    $data = json_decode($body, true);
    if (json_last_error() === JSON_ERROR_NONE) {
      return $data;
    }

    return new WP_Error('invalid_response', 'Invalid response from license server');
  }
}
