<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/public
 * @author     SEOLAXY
 */
class Seolaxy_Ctr_Public
{

  /**
   * The ID of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $plugin_name    The ID of this plugin.
   */
  private $plugin_name;

  /**
   * The version of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $version    The current version of this plugin.
   */
  private $version;

  /**
   * Unique ID for this class.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $id    The ID of this class.
   */
  private $id;

  private $seolaxy_ctr_options;

  private $logger;
  private $content_type_service;
  private $pattern_service;
  private $schema_service;
  private $meta_service;
  private $title_service;
  private $opengraph_service;

  /**
   * Admin notices
   *
   * @var array
   */
  private $admin_notices = array();

  /**
   * Initialize the class and set its properties.
   *
   * @since    1.0.0
   * @param      string $plugin_name       The name of the plugin.
   * @param      string $version    The version of this plugin.
   */
  public function __construct($plugin_name, $version)
  {

    $this->plugin_name = $plugin_name;
    $this->version = $version;
    $this->id = $this->plugin_name . '-public';

    $this->seolaxy_ctr_options = get_option('seolaxy_ctr_options');

    // Initialize services
    $this->init_services();

    // Add hooks
    add_action('admin_notices', array($this, 'display_admin_notices'));
    add_action('current_screen', array($this, 'check_dependencies'));
  }

  /**
   * Initialize services
   */
  private function init_services()
  {
    // Load base services first
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-logger-service.php';
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-pattern-service.php';
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-content-type-service.php';

    // Initialize base services
    $this->logger = new Seolaxy_Ctr_Logger_Service();
    $this->content_type_service = new Seolaxy_Ctr_Content_Type_Service($this->logger);
    $this->pattern_service = new Seolaxy_Ctr_Pattern_Service($this->seolaxy_ctr_options);

    // Load dependent services
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-schema-service.php';
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-meta-service.php';
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-title-service.php';
    require_once SEOLAXY_CTR_PATH . 'includes/services/class-seolaxy-ctr-opengraph-service.php';

    // Initialize dependent services
    $this->schema_service = new Seolaxy_Ctr_Schema_Service($this->seolaxy_ctr_options, $this->logger);
    $this->meta_service = new Seolaxy_Ctr_Meta_Service($this->seolaxy_ctr_options, $this->logger, $this->pattern_service);
    $this->title_service = new Seolaxy_Ctr_Title_Service($this->seolaxy_ctr_options, $this->logger, $this->pattern_service);
    $this->opengraph_service = new Seolaxy_Ctr_Opengraph_Service(
      $this->seolaxy_ctr_options,
      $this->logger,
      $this->title_service,
      $this->meta_service
    );
  }

  /**
   * Register the JavaScript and stylesheets for the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function enqueue_public_resources()
  {
    /**
     * This function is provided for demonstration purposes only.
     *
     * An instance of this class should be passed to the run() function
     * defined in Seolaxy_Ctr_Loader as all of the hooks are defined
     * in that particular class.
     *
     * The Seolaxy_Ctr_Loader will then create the relationship
     * between the defined hooks and the functions defined in this
     * class.
     */
    wp_enqueue_style($this->id, SEOLAXY_CTR_URL . 'public/css/seolaxy-ctr-public.css', array(), $this->version, 'all');
    wp_enqueue_script($this->id, SEOLAXY_CTR_URL . 'public/js/seolaxy-ctr-public.js', array('jquery'), $this->version, false);
  }

  /**
   * Set schemas for the current page
   */
  public function set_schemas()
  {
    $this->content_type_service->maybe_process([$this->schema_service, 'generate_schemas']);
  }

  /**
   * Set meta description
   */
  public function set_meta_description()
  {
    $this->content_type_service->maybe_process(function () {
      try {
        $description = $this->meta_service->get_meta_description();
        if ($description) {
          echo '<meta name="description" content="' . esc_attr($description) . '">' . "\n";
        }
      } catch (Exception $e) {
        $this->logger->error('Error setting meta description: ' . $e->getMessage());
      }
    });
  }

  /**
   * Set title tag parts
   */
  public function set_title_tag($title_parts)
  {
    return $this->content_type_service->maybe_process(
      [$this->title_service, 'filter_title_parts'],
      $title_parts
    ) ?? $title_parts;
  }

  /**
   * Set custom favicon
   */
  public function set_custom_favicon()
  {
    $this->content_type_service->maybe_process(function () {
      if (isset($this->seolaxy_ctr_options['favicon'])) {
        echo '<link rel="shortcut icon" href="' . esc_url($this->seolaxy_ctr_options['favicon']) . '" />';
      }
    });
  }

  /**
   * Set OpenGraph meta tags
   */
  public function set_open_graph_meta()
  {
    $this->content_type_service->maybe_process([$this->opengraph_service, 'output_opengraph_tags']);
  }

  function set_post_rich_snippet() {}

  /**
   * Check plugin dependencies
   */
  public function check_dependencies($screen)
  {
    // Only check on plugin pages
    if (!$screen || strpos($screen->base, 'toplevel_page_seolaxy-ctr') === false) {
      return;
    }

    // Check if Seolaxy CTR Woo is active
    if (class_exists('WooCommerce') && !class_exists('Seolaxy_Ctr_Woo')) {
      $this->admin_notices[] = array(
        'type' => 'info',
        'message' => sprintf(
          /* translators: %1$s: Plugin name, %2$s: Download link */
          __('The %1$s plugin is used for WordPress posts and pages. <a href="%2$s" target="_blank">Download SEOLAXY CTR Woo</a> for WooCommerce.', 'seolaxy-ctr'),
          '<strong>SEOLAXY CTR</strong>',
          'http://plugins-cdn.seolaxy.com/ctr-woo/latest.zip'
        )
      );
    }
  }

  /**
   * Display admin notices
   */
  public function display_admin_notices()
  {
    foreach ($this->admin_notices as $notice) {
      if (function_exists('wp_admin_notice')) {
        wp_admin_notice(
          $notice['message'],
          array(
            'type' => $notice['type'],
            'paragraph_wrap' => true,
            'is_dismissible' => false,
          )
        );
      } else {
        // Fallback for WordPress versions < 6.4
        printf(
          '<div class="notice notice-%1$s is-dismissible"><p>%2$s</p></div>',
          esc_attr($notice['type']),
          wp_kses_post($notice['message'])
        );
      }
    }
  }
}
