<?php

/**
 * Handles all Schema.org related functionality
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes/services
 */
class Seolaxy_Ctr_Schema_Service
{
    private $options;
    private $logger;

    public function __construct($options, $logger)
    {
        $this->options = $options;
        $this->logger = $logger;
    }

    /**
     * Generate and output schema markup
     */
    public function generate_schemas()
    {
        try {
            $website_type = $this->options['type_choose'] ?? '';
            $author_rich_snippet_post = $this->options['author_rich_shippet_post'] ?? '';
            $author_rich_snippet_page = $this->options['author_rich_shippet_page'] ?? '';

            // Generate main schema only on homepage when not WooCommerce shop
            if ($this->should_show_main_schema()) {
                switch ($website_type) {
                    case 'company':
                        $main_schema = $this->get_organization_schema();
                        $this->output_schema($main_schema);
                        break;

                    case 'personal':
                        $main_schema = $this->get_personal_schema();
                        $this->output_schema($main_schema);
                        break;

                    case 'business':
                        // Generate both organization and personal schemas
                        $organization_schema = $this->get_organization_schema();
                        $personal_schema = $this->get_personal_schema();
                        $this->output_schema($organization_schema);
                        $this->output_schema($personal_schema);
                        break;

                    case 'news':
                        $main_schema = $this->get_news_organization_schema();
                        $this->output_schema($main_schema);
                        break;

                    default:
                        // Fallback to personal schema for unknown types
                        $main_schema = $this->get_personal_schema();
                        $this->output_schema($main_schema);
                        break;
                }
            }

            // Generate post/page schema if needed
            if (
                !$this->is_woocommerce_content() &&
                ((is_single() && $author_rich_snippet_post) ||
                    (is_page() && $author_rich_snippet_page))
            ) {
                $post_schema = $this->get_post_schema();
                $this->output_schema($post_schema);
            }
        } catch (Exception $e) {
            $this->logger->error('Error generating schemas: ' . $e->getMessage());
        }
    }

    /**
     * Output schema data as JSON-LD
     */
    private function output_schema($schema_data)
    {
        if (empty($schema_data)) {
            $this->logger->warning('Empty schema data detected');
            return;
        }

        echo '<script type="application/ld+json">'
            . wp_json_encode($schema_data, JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT)
            . '</script>';
    }

    /**
     * Generate organization blog schema
     */
    private function get_organization_schema()
    {
        // Base schema structure - same as personal schema
        $schema_data = [
            '@context' => 'https://schema.org',
            '@type' => 'Blog',
            'url' => home_url(),
        ];

        // Add blog name with same fallback logic as personal schema
        if (!empty($this->options['website_name'])) {
            $schema_data['name'] = $this->options['website_name'];
        } else {
            $schema_data['name'] = get_bloginfo('name');
        }

        // Add blog description with same fallback as personal schema
        if (!empty($this->options['blog_description'])) {
            $schema_data['description'] = $this->options['blog_description'];
        } else {
            $schema_data['description'] = get_bloginfo('description');
        }

        // Build publisher (Organization) data
        $publisher = [
            '@type' => 'Organization',
            'url' => home_url()
        ];

        // Add organization name
        if (!empty($this->options['company_name'])) {
            $publisher['name'] = $this->options['company_name'];
        }

        // Add logo if exists
        if (!empty($this->options['organization_logo'])) {
            $publisher['logo'] = [
                '@type' => 'ImageObject',
                'url' => $this->options['organization_logo']
            ];
        }

        // Add address if exists
        $address = $this->get_address_data();
        if (!empty($address)) {
            $publisher['address'] = $address;
        }

        // Add social media profiles if they exist
        $sameAs = array_filter([
            $this->options['social_profile_1'] ?? '',
            $this->options['social_profile_2'] ?? ''
        ]);

        if (!empty($sameAs)) {
            $publisher['sameAs'] = array_values($sameAs);
        }

        // Add brand if exists
        if (!empty($this->options['brand_name'])) {
            $publisher['brand'] = [
                '@type' => 'Brand',
                'name' => $this->options['brand_name']
            ];
        }

        // Add publisher to main schema
        $schema_data['publisher'] = $publisher;

        return $schema_data;
    }

    /**
     * Generate personal blog schema
     */
    private function get_personal_schema()
    {
        // Base schema structure
        $schema_data = [
            '@context' => 'https://schema.org',
            '@type' => 'Blog',
            'url' => home_url(),
        ];

        // Add blog name - use website name or fallback to person's name + "'s Blog"
        if (!empty($this->options['website_name'])) {
            $schema_data['name'] = $this->options['website_name'];
        } elseif (!empty($this->options['first_and_last_name'])) {
            $schema_data['name'] = $this->options['first_and_last_name'] . "'s Blog";
        } else {
            $schema_data['name'] = get_bloginfo('name');
        }

        // Add blog description
        if (!empty($this->options['blog_description'])) {
            $schema_data['description'] = $this->options['blog_description'];
        } else {
            $schema_data['description'] = get_bloginfo('description');
        }

        // Build publisher (Person) data
        $publisher = [
            '@type' => 'Person',
        ];

        // Add person's name
        if (!empty($this->options['first_and_last_name'])) {
            $publisher['name'] = $this->options['first_and_last_name'];
        }

        // Add person's URL
        $publisher['url'] = home_url();

        // Add social media profiles if they exist
        $sameAs = array_filter([
            $this->options['social_profile_1'] ?? '',
            $this->options['social_profile_2'] ?? ''
        ]);

        if (!empty($sameAs)) {
            $publisher['sameAs'] = array_values($sameAs);
        }

        // Add publisher to main schema
        $schema_data['publisher'] = $publisher;

        return $schema_data;
    }

    /**
     * Generate news media organization schema
     */
    private function get_news_organization_schema()
    {
        // Base schema structure
        $schema_data = [
            '@context' => 'https://schema.org',
            '@type' => 'NewsMediaOrganization',
            'url' => home_url(),
        ];

        // Add organization name
        if (!empty($this->options['website_name'])) {
            $schema_data['name'] = $this->options['website_name'];
        } else {
            $schema_data['name'] = get_bloginfo('name');
        }

        // Add organization description
        if (!empty($this->options['blog_description'])) {
            $schema_data['description'] = $this->options['blog_description'];
        } else {
            $schema_data['description'] = get_bloginfo('description');
        }

        // Add logo if exists
        if (!empty($this->options['organization_logo'])) {
            $schema_data['logo'] = [
                '@type' => 'ImageObject',
                'url' => $this->options['organization_logo']
            ];
        }

        // Add address if exists
        $address = $this->get_address_data();
        if (!empty($address)) {
            $schema_data['address'] = $address;
        }

        // Add contact point if exists
        $contact_point = $this->get_contact_point_data();
        if (!empty($contact_point)) {
            $schema_data['contactPoint'] = $contact_point;
        }

        // Add social media profiles if they exist
        $sameAs = array_filter([
            $this->options['social_profile_1'] ?? '',
            $this->options['social_profile_2'] ?? ''
        ]);

        if (!empty($sameAs)) {
            $schema_data['sameAs'] = array_values($sameAs);
        }

        // Add brand if exists
        if (!empty($this->options['brand_name'])) {
            $schema_data['brand'] = [
                '@type' => 'Brand',
                'name' => $this->options['brand_name']
            ];
        }

        return $schema_data;
    }

    /**
     * Generate post/page schema based on website type
     */
    private function get_post_schema()
    {
        // Don't show on homepage
        if (is_front_page() || is_home()) {
            return [];
        }

        // Base schema structure
        $schema_data = [
            '@context' => 'https://schema.org',
            '@type' => 'BlogPosting',
            'headline' => get_the_title(),
            'url' => get_permalink(),
            'datePublished' => get_the_date('c'),
            'description' => get_post_meta(get_the_ID(), 'seolaxy_ctr_post_page_meta_description_content', true)
        ];

        // Add publisher based on website type
        $website_type = $this->options['type_choose'] ?? '';

        if ($website_type === 'company') {
            $schema_data['publisher'] = $this->get_organization_publisher();
        } else {
            $schema_data['author'] = $this->get_personal_author();
        }

        return $schema_data;
    }

    /**
     * Get organization publisher data for post schema
     */
    private function get_organization_publisher()
    {
        $publisher = [
            '@type' => 'Organization',
            'name' => $this->options['website_name'] ?? get_bloginfo('name'),
            'url' => home_url()
        ];

        // Add logo if exists
        if (!empty($this->options['organization_logo'])) {
            $publisher['logo'] = [
                '@type' => 'ImageObject',
                'url' => $this->options['organization_logo']
            ];
        }

        // Add contact point if exists
        $contact_point = $this->get_contact_point_data();
        if (!empty($contact_point)) {
            $publisher['contactPoint'] = $contact_point;
        }

        // Add address if exists
        $address = $this->get_address_data();
        if (!empty($address)) {
            $publisher['address'] = $address;
        }

        // Add social media profiles if they exist
        $sameAs = array_filter([
            $this->options['social_profile_1'] ?? '',
            $this->options['social_profile_2'] ?? ''
        ]);

        if (!empty($sameAs)) {
            $publisher['sameAs'] = array_values($sameAs);
        }

        return $publisher;
    }

    /**
     * Get personal author data for post schema
     */
    private function get_personal_author()
    {
        global $post;
        $author_id = $post->post_author;
        $author = [
            '@type' => 'Person',
            'name' => get_the_author_meta('display_name', $author_id),
            'url' => $this->get_author_url($author_id)
        ];

        // Add author image if exists
        $avatar_url = get_avatar_url($author_id);
        if ($avatar_url) {
            $author['image'] = $avatar_url;
        }

        // Add social media profiles if they exist
        $sameAs = array_filter([
            $this->options['social_profile_1'] ?? '',
            $this->options['social_profile_2'] ?? ''
        ]);

        if (!empty($sameAs)) {
            $author['sameAs'] = array_values($sameAs);
        }

        return $author;
    }

    /**
     * Get author URL based on settings
     */
    private function get_author_url($author_id)
    {
        return $this->options['linked_author_profile'] === 'external'
            ? get_the_author_meta('url', $author_id)
            : get_author_posts_url($author_id);
    }

    /**
     * Get contact point data
     */
    private function get_contact_point_data()
    {
        $email = $this->options['email_address'] ?? '';
        $phone = $this->options['phone'] ?? '';

        if (empty($email) && empty($phone)) {
            return null;
        }

        $contact_point = [
            '@type' => 'ContactPoint',
            'contactType' => 'customer service'
        ];

        if (!empty($phone)) {
            $contact_point['telephone'] = $phone;
        }

        if (!empty($email)) {
            $contact_point['email'] = $email;
        }

        return $contact_point;
    }

    /**
     * Get address data
     */
    private function get_address_data()
    {
        $address_fields = [
            'company_address' => 'streetAddress',
            'city' => 'addressLocality',
            'state' => 'addressRegion',
            'postal_code' => 'postalCode',
            'country' => 'addressCountry'
        ];

        $address = ['@type' => 'PostalAddress'];
        $has_data = false;

        foreach ($address_fields as $option_key => $schema_key) {
            if (!empty($this->options[$option_key])) {
                $address[$schema_key] = $this->options[$option_key];
                $has_data = true;
            }
        }

        return $has_data ? $address : null;
    }

    /**
     * Check if main schema should be shown
     */
    private function should_show_main_schema()
    {
        // Only show on homepage
        if (!is_front_page()) {
            return false;
        }

        // Check if WooCommerce is active and if homepage is shop
        if (class_exists('WooCommerce') && get_option('woocommerce_shop_page_id') === get_option('page_on_front')) {
            return false;
        }

        return true;
    }

    /**
     * Check if current post/page is a WooCommerce content
     */
    private function is_woocommerce_content()
    {
        // Check if WooCommerce is active
        if (!class_exists('WooCommerce')) {
            return false;
        }

        // Check for WooCommerce pages
        if (is_woocommerce() || is_cart() || is_checkout() || is_account_page()) {
            return true;
        }

        // Check if current post is a product
        if (get_post_type() === 'product') {
            return true;
        }

        return false;
    }
}
