<?php

/**
 * Handles pattern replacements for titles and meta descriptions
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes/services
 */
class Seolaxy_Ctr_Pattern_Service {
    private $options;

    public function __construct($options) {
        $this->options = $options;
    }

    /**
     * Replace all available patterns in a string
     *
     * @param string $text Text containing patterns to replace
     * @return string Text with replaced patterns
     */
    public function replace_patterns($text) {
        if (empty($text)) {
            return '';
        }

        $patterns = [
            '{{post_title}}' => $this->get_post_title(),
            '{{page_title}}' => $this->get_post_title(),
            '{{author_name}}' => $this->get_author_name(),
            '{{current_year}}' => date('Y'),
            '{{site_name}}' => get_bloginfo('name'),
            '{{separator}}' => $this->options['separator'] ?? '',
            '{{tag_name}}' => $this->get_tag_name(),
            '{{category_name}}' => $this->get_category_name()
        ];

        return str_replace(
            array_keys($patterns),
            array_values($patterns),
            $text
        );
    }

    /**
     * Get current post/page title
     */
    private function get_post_title() {
        return get_the_title() ?: '';
    }

    /**
     * Get current author name
     */
    private function get_author_name() {
        global $post;
        if (!$post) {
            return '';
        }
        
        return get_the_author_meta('display_name', $post->post_author) ?: '';
    }

    /**
     * Get current tag name
     */
    private function get_tag_name() {
        if (!is_tag()) {
            return '';
        }

        $tag = get_queried_object();
        return $tag->name ?? '';
    }

    /**
     * Get current post category name
     * Returns comma-separated category names for posts, empty for other post types
     */
    private function get_category_name() {
        // Only for single posts
        if (!is_single() || get_post_type() !== 'post') {
            return '';
        }

        $categories = get_the_category();
        if (!empty($categories)) {
            // Get all category names
            $category_names = array_map(function($category) {
                return $category->name;
            }, $categories);

            // Return comma-separated category names
            return implode(', ', $category_names);
        }

        return '';
    }
} 