<?php

/**
 * Handles all OpenGraph related functionality
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes/services
 */
class Seolaxy_Ctr_Opengraph_Service {
    private $options;
    private $logger;
    private $title_service;
    private $meta_service;

    public function __construct($options, $logger, $title_service, $meta_service) {
        $this->options = $options;
        $this->logger = $logger;
        $this->title_service = $title_service;
        $this->meta_service = $meta_service;
    }

    /**
     * Output all OpenGraph meta tags
     */
    public function output_opengraph_tags() {
        try {
            $this->output_title();
            $this->output_description();
            $this->output_image();
            $this->output_type();
        } catch (Exception $e) {
            $this->logger->error('Error outputting OpenGraph tags: ' . $e->getMessage());
        }
    }

    /**
     * Output OpenGraph title
     */
    private function output_title() {
        $title = $this->title_service->get_title_tag();
        if ($title) {
            echo '<meta property="og:title" content="' . esc_attr($title) . '">' . "\n";
        }
    }

    /**
     * Output OpenGraph description
     */
    private function output_description() {
        $description = $this->meta_service->get_meta_description();
        if ($description) {
            echo '<meta property="og:description" content="' . esc_attr($description) . '">' . "\n";
        }
    }

    /**
     * Output OpenGraph image
     */
    private function output_image() {
        $image_url = $this->get_og_image();
        if ($image_url) {
            echo '<meta property="og:image" content="' . esc_url($image_url) . '">' . "\n";
        }
    }

    /**
     * Get OpenGraph image URL
     */
    private function get_og_image() {
        if (is_home() || is_front_page()) {
            return $this->options['featured_image'] ?? '';
        }

        if (is_single() || is_page()) {
            return get_the_post_thumbnail_url(get_the_ID(), 'full');
        }

        return '';
    }

    /**
     * Output OpenGraph type
     */
    private function output_type() {
        $type = $this->get_og_type();
        echo '<meta property="og:type" content="' . esc_attr($type) . '">' . "\n";
    }

    /**
     * Get OpenGraph type based on page context
     */
    private function get_og_type() {
        if (is_single() || is_page()) {
            return 'article';
        }
        return 'website';
    }
} 