<?php

/**
 * Handles all meta tags related functionality
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes/services
 */
class Seolaxy_Ctr_Meta_Service {
    private $options;
    private $logger;
    private $pattern_service;

    public function __construct($options, $logger, $pattern_service) {
        $this->options = $options;
        $this->logger = $logger;
        $this->pattern_service = $pattern_service;
    }

    /**
     * Get meta description based on current page context
     */
    public function get_meta_description() {
        try {
            if (is_home() || is_front_page()) {
                return $this->options['homepage_meta_description'] ?? '';
            }

            if (is_category()) {
                return $this->get_category_meta_description();
            }

            if (is_single()) {
                return $this->get_single_post_page_description('post_meta_description', 'post_meta_description_empty');
            }

            if (is_page()) {
                return $this->get_single_post_page_description('page_meta_description', 'page_meta_description_empty');
            }

            if (is_tag()) {
                return $this->get_tag_meta_description();
            }

            return '';
        } catch (Exception $e) {
            $this->logger->error('Error getting meta description: ' . $e->getMessage());
            return '';
        }
    }

    /**
     * Get meta description for single posts and pages
     */
    private function get_single_post_page_description($meta_key, $empty_key) {
        $content_type = $this->options[$meta_key] ?? '';
        
        if (empty($content_type)) {
            return '';
        }

        $description = $this->get_primary_description($content_type);
        if (empty($description)) {
            $description = $this->get_fallback_description($empty_key);
        }

        return $description;
    }

    /**
     * Get primary description based on content type
     */
    private function get_primary_description($content_type) {
        switch ($content_type) {
            case 'in-post-meta-description':
                return get_post_meta(get_the_ID(), 'seolaxy_ctr_post_page_meta_description_content', true);
            
            case 'post-intro':
                return $this->get_cleaned_content_excerpt();
            
            case 'excerpt':
                return has_excerpt() ? strip_tags(get_the_excerpt()) : '';
            
            default:
                return '';
        }
    }

    /**
     * Get fallback description when primary is empty
     */
    private function get_fallback_description($empty_key) {
        $fallback_type = $this->options[$empty_key] ?? '';
        
        switch ($fallback_type) {
            case 'post-intro':
                $content_without_tags = strip_tags(get_the_content());
                $clean_content = str_replace(array("\r\n", "\n", "\r"), '', $content_without_tags);
                return substr($clean_content, 0, 155);
            
            case 'excerpt':
                return has_excerpt() ? strip_tags(get_the_excerpt()) : '';
            
            case 'in-post-meta-description':
                return get_post_meta(get_the_ID(), 'seolaxy_ctr_post_page_meta_description_content', true);
            
            default:
                return '';
        }
    }

    /**
     * Get cleaned content excerpt
     */
    private function get_cleaned_content_excerpt() {
        $content = strip_tags(get_the_content());
        $content = str_replace(["\r\n", "\n", "\r"], '', $content);
        return substr($content, 0, 155);
    }

    /**
     * Get category meta description
     */
    private function get_category_meta_description() {
        $queried_object = get_queried_object();
        if (!$queried_object) {
            return '';
        }

        $category_slug = $queried_object->slug;
        return $this->options['category_meta'][$category_slug]['meta_description'] ?? '';
    }

    /**
     * Get tag meta description
     */
    private function get_tag_meta_description() {
        $pattern = $this->options['tag_meta_description_pattern'] ?? '';
        return $this->pattern_service->replace_patterns($pattern);
    }
} 