<?php

/**
 * Service to determine if the plugin should process the current content
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes/services
 */
class Seolaxy_Ctr_Content_Type_Service
{
    private $logger;

    /**
     * Initialize the class
     *
     * @param Seolaxy_Ctr_Logger_Service $logger Logger service
     */
    public function __construct($logger)
    {
        $this->logger = $logger;
    }

    /**
     * Check if current content should be processed by this plugin
     * This should be called in every public hook callback
     *
     * @return bool True if plugin should process current content
     */
    public function should_process(): bool
    {
        // Bail early if we're in admin
        if (is_admin()) {
            return false;
        }

        // Check for homepage/front page
        if (is_front_page() || is_home()) {
            // If WooCommerce is active, check if shop is set as homepage
            if (function_exists('is_shop') && is_shop()) {
                $this->logger->debug('Skipping content: WooCommerce shop page set as homepage');
                return false;
            }
            return true;
        }

        // Get current post type
        $post_type = get_post_type();

        // Check for default WordPress post types
        if (in_array($post_type, ['post', 'page'])) {
            return true;
        }

        // Check for default WordPress taxonomies
        if (is_category() || is_tag()) {
            return true;
        }

        return false;
    }

    /**
     * Helper method to wrap callbacks with content type checking
     *
     * @param callable $callback The callback to wrap
     * @param mixed ...$args The arguments to pass to the callback
     * @return mixed|null The callback result or null if content should be skipped
     */
    public function maybe_process(callable $callback, ...$args)
    {
        if (!$this->should_process()) {
            return null;
        }

        return call_user_func_array($callback, $args);
    }
}
