<?php

/**
 * Get the Plugin Default Options.
 *
 * @since 1.0.0
 *
 * @param null
 *
 * @return array Default Options
 *
 * @author     SEOLAXY
 *
 */
if (!function_exists('seolaxy_ctr_default_options')) :
  function seolaxy_ctr_default_options()
  {
    // Get current user info for name
    $current_user = wp_get_current_user();
    $user_name = '';
    if ($current_user->exists()) {
      $first_name = get_user_meta($current_user->ID, 'first_name', true);
      $last_name = get_user_meta($current_user->ID, 'last_name', true);
      if (!empty($first_name) || !empty($last_name)) {
        $user_name = trim($first_name . ' ' . $last_name);
      }
    }

    $default_theme_options = array(
      'type_choose' => esc_html('personal'),
      'first_and_last_name' => esc_html($user_name),
      'website_name' => esc_html(get_option('blogname', '')),
      'brand_name' => esc_html(''),
      'company_name' => esc_html(''),
      'company_address' => esc_html(''),
      'country' => esc_html(''),
      'state' => esc_html(''),
      'postal_code' => esc_html(''),
      'city' => esc_html(''),
      'email_address' => esc_html(get_option('admin_email', '')),
      'phone' => esc_html(''),
      'social_profile_1' => esc_html(''),
      'social_profile_2' => esc_html(''),
      'blog_description' => esc_html(get_option('blogdescription', '')),
      'organization_logo' => esc_html(''),
      'homepage_title_tag' => esc_html(''),
      'homepage_meta_description' => esc_html(''),
      'favicon' => esc_html(''),
      'featured_image' => esc_html(''),
      'separator' => esc_html('|'),
      'post_title_tag_pattern' => esc_html('{{post_title}} by {{author_name}} {{separator}} {{site_name}}'),
      'post_title_tag_pattern_recommended' => true,
      'post_meta_description' => esc_html('excerpt'),
      'post_meta_description_empty' => esc_html('in-post-meta-description'),
      'post_meta_description_preview' => false,

      'page_title_tag_pattern' => esc_html('{{page_title}} by {{author_name}} {{separator}} {{site_name}}'),
      'page_title_tag_pattern_recommended' => false,
      'page_meta_description' => esc_html('excerpt'),
      'page_meta_description_empty' => esc_html('in-post-meta-description'),
      'page_meta_description_preview' => false,

      'tag_title_tag_pattern' => esc_html('{{tag_name}} {{current_year}} {{separator}} {{site_name}}'),
      'tag_title_tag_pattern_recommended' => false,
      'tag_meta_description_pattern' => esc_html('See a collection of all recent articles about {{tag_name}} in {{current_year}} and all archived articles published by {{site_name}} about {{tag_name}}'),
      'tag_meta_description_pattern_recommended' => false,
      'tag_meta_description_preview' => false,

      'author_rich_shippet_post' => false,
      'author_rich_shippet_page' => false,
      'linked_author_profile' => esc_html('external'),
    );

    return apply_filters('seolaxy_ctr_default_options', $default_theme_options);
  }
endif;

/**
 * Get the Plugin Saved Options.
 *
 * @since 1.0.0
 *
 * @param string $key optional option key
 *
 * @return mixed All Options Array Or Options Value
 *
 * @author     SEOLAXY
 *
 */
if (!function_exists('seolaxy_ctr_get_options')) :
  function seolaxy_ctr_get_options($key = '')
  {
    $options = get_option('seolaxy_ctr_options');
    $default_options = seolaxy_ctr_default_options();

    if (!empty($key)) {
      if (isset($options[$key])) {
        return $options[$key];
      }
      return isset($default_options[$key]) ? $default_options[$key] : false;
    } else {
      if (!is_array($options)) {
        $options = array();
      }
      return array_merge($default_options, $options);
    }
  }
endif;

function seolaxy_ctr_fix_schema($option_group, $option_name, $args)
{
  if ($option_name === 'seolaxy_ctr_options') {
    $my_options = get_option($option_name, $args['default']);
    $my_options_keys = array_keys($my_options);

    $schema_properties = $args['show_in_rest']['schema']['properties'];
    $schema_properties_keys = array_keys($schema_properties);

    $different_keys = array_diff($my_options_keys, $schema_properties_keys);
    if ($different_keys) {
      foreach ($different_keys as $key) {
        unset($my_options[$key]);
      }
      update_option($option_name, $my_options);
    }
  }
}
add_action('register_setting', 'seolaxy_ctr_fix_schema', 10, 3);

//POST / PAGE meta
function seolaxy_ctr_post_page_meta_box()
{
  add_meta_box(
    // Unique ID
    'seolaxy_ctr_post_page_meta_description',
    // Box title
    'SEOLAXY CTR - Meta Description',
    // Content callback
    'seolaxy_ctr_post_page_meta_description_field',
    // Post types
    ['post', 'page'],
    // Context
    'normal',
    'high' // Priority
  );
}
add_action('add_meta_boxes', 'seolaxy_ctr_post_page_meta_box');

function seolaxy_ctr_post_page_meta_description_field($post)
{
  $meta_value = get_post_meta($post->ID, 'seolaxy_ctr_post_page_meta_description_content', true);
  wp_nonce_field(plugin_basename(__FILE__), 'seolaxy_ctr_post_page_meta_nonce');
  echo '<textarea name="seolaxy_ctr_post_page_meta_field" rows="10" style="width: 100%;">' . esc_textarea($meta_value) . '</textarea>';
}

function seolaxy_ctr_post_page_meta_save($post_id)
{
  // Check for nonce for security
  if (!isset($_POST['seolaxy_ctr_post_page_meta_nonce']) || !wp_verify_nonce($_POST['seolaxy_ctr_post_page_meta_nonce'], plugin_basename(__FILE__))) {
    return;
  }

  if (array_key_exists('seolaxy_ctr_post_page_meta_field', $_POST)) {
    update_post_meta($post_id, 'seolaxy_ctr_post_page_meta_description_content', $_POST['seolaxy_ctr_post_page_meta_field']);
  }
}
add_action('save_post', 'seolaxy_ctr_post_page_meta_save');

add_action('rest_api_init', 'seolaxy_ctr_register_post_meta');

function seolaxy_ctr_register_post_meta()
{
  register_rest_field(
    'post',
    'seolaxy_ctr_post_page_meta_description_content',
    array(
      'get_callback' => 'seolaxy_ctr_get_post_meta_for_api',
      'schema' => null,
    )
  );
}

function seolaxy_ctr_get_post_meta_for_api($object)
{
  // Get the id of the post object array
  $post_id = $object['id'];

  // Return the post meta
  return get_post_meta($post_id, 'seolaxy_ctr_post_page_meta_description_content', true);
}
