<?php

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Seolaxy_Ctr
 * @subpackage Seolaxy_Ctr/includes
 * @author     SEOLAXY
 */
class Seolaxy_Ctr
{

  /**
   * The loader that's responsible for maintaining and registering all hooks that power
   * the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      Seolaxy_Ctr_Loader    $loader    Maintains and registers all hooks for the plugin.
   */
  protected $loader;

  /**
   * The unique identifier of this plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $plugin_name    The string used to uniquely identify this plugin.
   */
  protected $plugin_name;

  /**
   * The current version of the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $version    The current version of the plugin.
   */
  protected $version;

  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct()
  {
    $this->version = SEOLAXY_CTR_VERSION;
    $this->plugin_name = 'seolaxy-ctr';

    $this->load_dependencies();
    $this->set_locale();
    $this->define_include_hooks();
    $this->define_admin_hooks();
    $this->define_public_hooks();
  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Include the following files that make up the plugin:
   *
   * - Seolaxy_Ctr_Loader. Orchestrates the hooks of the plugin.
   * - Seolaxy_Ctr_i18n. Defines internationalization functionality.
   * - Seolaxy_Ctr_Admin. Defines all hooks for the admin area.
   * - Seolaxy_Ctr_Public. Defines all hooks for the public side of the site.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function load_dependencies()
  {

    /**
     * Plugin Core Functions.
     */
    require_once SEOLAXY_CTR_PATH . 'includes/functions.php';

    /**
     * The class responsible for orchestrating the actions and filters of the
     * core plugin.
     */
    require_once SEOLAXY_CTR_PATH . 'includes/class-seolaxy-ctr-loader.php';

    /**
     * The class responsible for defining internationalization functionality
     * of the plugin.
     */
    require_once SEOLAXY_CTR_PATH . 'includes/class-seolaxy-ctr-i18n.php';

    /**
     * The class responsible for defining all actions that occur in both admin and public-facing areas.
     */
    require_once SEOLAXY_CTR_PATH . 'includes/class-seolaxy-ctr-include.php';

    /**
     * The class responsible for defining all actions that occur in the admin area.
     */
    require_once SEOLAXY_CTR_PATH . 'admin/class-seolaxy-ctr-admin.php';

    /**
     * The class responsible for defining all actions that occur in the public-facing
     * side of the site.
     */
    require_once SEOLAXY_CTR_PATH . 'public/class-seolaxy-ctr-public.php';

    $this->loader = new Seolaxy_Ctr_Loader();
  }

  /**
   * Define the locale for this plugin for internationalization.
   *
   * Uses the Seolaxy_Ctr_i18n class in order to set the domain and to register the hook
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function set_locale()
  {

    $plugin_i18n = new Seolaxy_Ctr_i18n();

    $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
  }

  /**
   * Register all of the hooks related to both admin and public-facing areas functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_include_hooks()
  {

    $plugin_admin = new Seolaxy_Ctr_Include($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('init', $plugin_admin, 'init_something');
  }

  /**
   * Register all of the hooks related to the admin area functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_admin_hooks()
  {

    $plugin_admin = new Seolaxy_Ctr_Admin($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('admin_menu', $plugin_admin, 'add_admin_menu');
    $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_resources');

    /*Register Settings*/
    $this->loader->add_action('rest_api_init', $plugin_admin, 'register_settings');
    $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');
  }

  /**
   * Register all of the hooks related to the public-facing functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_public_hooks()
  {

    $plugin_public = new Seolaxy_Ctr_Public($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_public_resources');
    $this->loader->add_action('wp_head', $plugin_public, 'set_schemas');
    $this->loader->add_filter('wp_head', $plugin_public, 'set_meta_description');
    $this->loader->add_filter('wp_head', $plugin_public, 'set_custom_favicon');
    $this->loader->add_filter('document_title_parts', $plugin_public, 'set_title_tag');
    $this->loader->add_filter('wp_head', $plugin_public, 'set_open_graph_meta');
  }

  /**
   * Run the loader to execute all of the hooks with WordPress.
   *
   * @since    1.0.0
   */
  public function run()
  {
    $this->loader->run();
  }

  /**
   * The name of the plugin used to uniquely identify it within the context of
   * WordPress and to define internationalization functionality.
   *
   * @since     1.0.0
   * @return    string    The name of the plugin.
   */
  public function get_plugin_name()
  {
    return $this->plugin_name;
  }

  /**
   * The reference to the class that orchestrates the hooks with the plugin.
   *
   * @since     1.0.0
   * @return    Seolaxy_Ctr_Loader    Orchestrates the hooks of the plugin.
   */
  public function get_loader()
  {
    return $this->loader;
  }

  /**
   * Retrieve the version number of the plugin.
   *
   * @since     1.0.0
   * @return    string    The version number of the plugin.
   */
  public function get_version()
  {
    return $this->version;
  }
}
