<?php

/**
 * The public-facing functionality of the plugin.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Crawler_Budget_Control
 * @subpackage Seolaxy_Crawler_Budget_Control/public
 */

/**
 * The public-facing functionality of the plugin.
 *
 * Defines the plugin name, version, and two examples hooks for how to
 * enqueue the public-facing stylesheet and JavaScript.
 *
 * @package    Seolaxy_Crawler_Budget_Control
 * @subpackage Seolaxy_Crawler_Budget_Control/public
 * @author     SEOLAXY
 */
class Seolaxy_Crawler_Budget_Control_Public
{

  /**
   * The ID of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $plugin_name    The ID of this plugin.
   */
  private $plugin_name;

  /**
   * The version of this plugin.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $version    The current version of this plugin.
   */
  private $version;

  /**
   * Unique ID for this class.
   *
   * @since    1.0.0
   * @access   private
   * @var      string    $id    The ID of this class.
   */
  private $id;
  private $seolaxy_crawler_budget_control_options;
  private $logger;

  /**
   * Initialize the class and set its properties.
   *
   * @since    1.0.0
   * @param      string $plugin_name       The name of the plugin.
   * @param      string $version    The version of this plugin.
   */
  public function __construct($plugin_name, $version)
  {

    $this->plugin_name = $plugin_name;
    $this->version = $version;
    $this->id = $this->plugin_name . '-public';

    $this->seolaxy_crawler_budget_control_options = get_option('seolaxy_crawler_budget_control_options');

    // Initialize logger
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'includes/services/class-seolaxy-crawler-budget-control-logger-service.php';
    $this->logger = new Seolaxy_Crawler_Budget_Control_Logger_Service();

    //In linkjuice plugin this option is set to default (10) it must be larger thah this
    add_action('wp_robots', array($this, 'robots'), 20);

    // Set up conflict resolution for other SEO plugins
    add_action('wp', array($this, 'setup_seo_conflicts_resolution'));
  }

  private function get_attribute_meta_values($attributes_array)
  {
    $attributes_meta_values = [];

    // Return empty array if WooCommerce is not active
    if (!seolaxy_crawler_budget_control_is_woocommerce_active()) {
      return $attributes_meta_values;
    }

    $attribute_taxonomies = wc_get_attribute_taxonomies();
    foreach ($attribute_taxonomies as $tax) {
      if (in_array($tax->attribute_name, $attributes_array)) {
        $seolaxy_noindex_attribute_value = get_option('seolaxy_noindex_attribute_' . $tax->attribute_id);
        $seolaxy_filterkombi_attribute_value = get_option('seolaxy_filterkombi_attribute_' . $tax->attribute_id);
        $tax->seolaxy_noindex_attribute_value = $seolaxy_noindex_attribute_value;
        $tax->seolaxy_filterkombi_attribute_value = $seolaxy_filterkombi_attribute_value;
        array_push($attributes_meta_values, $tax);
      }
    }
    
    // Log warning if no matches found but attributes_array is not empty
    if (empty($attributes_meta_values) && !empty($attributes_array)) {
      $this->logger->debug('get_attribute_meta_values() - WARNING: No matching attributes found. Looking for: ' . implode(', ', $attributes_array));
    }
    
    return $attributes_meta_values;
  }

  private function get_term_meta_values($filters)
  {
    $term_meta_values = [];
    foreach ($filters as $attribute => $attribute_terms) {
      $attribute_pa = 'pa_' . $attribute;

      foreach ($attribute_terms as $attribute_term) {
        $attribute_term_data = get_term_by('slug', $attribute_term, $attribute_pa);

        $attribute_term_data->attribute = $attribute;

        $seolaxy_noindex_term_value = get_term_meta($attribute_term_data->term_id, 'seolaxy_noindex_term', true);
        $attribute_term_data->seolaxy_noindex_term = $seolaxy_noindex_term_value;

        array_push($term_meta_values, $attribute_term_data);
      }
    }

    return $term_meta_values;
  }

  private function should_process_robots($term_count)
  {
    $attribute_terms_noindex_option = $this->get_option_value('attribute_terms_noindex', true);
    $attributes_noindex_option = $this->get_option_value('attributes_noindex', true);
    $attributes_filterkombi_option = $this->get_option_value('attributes_filterkombi', true);

    $options_enabled = ($attribute_terms_noindex_option || $attributes_noindex_option || $attributes_filterkombi_option);
    $term_count_valid = !($term_count === 0 || $term_count > 2);

    // If all options are disabled or term count is invalid, don't process
    if (
      !$options_enabled ||
      !$term_count_valid
    ) {
      return false;
    }

    return true;
  }

  private function get_option_value($key, $default = false)
  {
    return isset($this->seolaxy_crawler_budget_control_options[$key])
      ? $this->seolaxy_crawler_budget_control_options[$key]
      : $default;
  }

  private function process_single_attribute($term_meta_values, $attributes_meta_values)
  {
    $attribute_terms_noindex_option = $this->get_option_value('attribute_terms_noindex', true);
    $attributes_noindex_option = $this->get_option_value('attributes_noindex', true);

    if (
      count($term_meta_values) == 1 &&
      ($attribute_terms_noindex_option || $attributes_noindex_option)
    ) {

      $term_noindex = $term_meta_values[0]->seolaxy_noindex_term;
      
      // Fix: Check if attributes_meta_values is empty before accessing index 0
      $attribute_noindex = null;
      if (!empty($attributes_meta_values) && isset($attributes_meta_values[0])) {
        $attribute_noindex = $attributes_meta_values[0]->seolaxy_noindex_attribute_value;
      } else {
        // Fallback: Try to get attribute ID from term taxonomy and check option directly
        // This handles cases where get_attribute_meta_values() didn't find a match
        if (seolaxy_crawler_budget_control_is_woocommerce_active()) {
          $term = $term_meta_values[0];
          $attribute_name = isset($term->attribute) ? $term->attribute : null;
          
          if ($attribute_name) {
            $attribute_taxonomies = wc_get_attribute_taxonomies();
            foreach ($attribute_taxonomies as $tax) {
              if ($tax->attribute_name === $attribute_name) {
                $option_key = 'seolaxy_noindex_attribute_' . $tax->attribute_id;
                $attribute_noindex = get_option($option_key);
                $this->logger->debug('Fallback: Retrieved attribute option directly - attribute: ' . $attribute_name . ', option_key: ' . $option_key . ', value: ' . var_export($attribute_noindex, true));
                break;
              }
            }
          }
        }
      }

      if (($term_noindex && $attribute_terms_noindex_option) ||
        ($attribute_noindex && $attributes_noindex_option)
      ) {
        return ['noindex' => true, 'remove_index' => true];
      }
    }

    return null;
  }

  private function process_attribute_combination($attributes_meta_values)
  {
    $attributes_filterkombi_option = $this->get_option_value('attributes_filterkombi', true);

    if (count($attributes_meta_values) == 2 && $attributes_filterkombi_option) {
      if (
        $attributes_meta_values[0]->seolaxy_filterkombi_attribute_value &&
        $attributes_meta_values[1]->seolaxy_filterkombi_attribute_value
      ) {
        return ['index' => true, 'remove_noindex' => true];
      }
    }

    return null;
  }

  private function transform_chosen_attributes_to_filter_data($chosen_attributes)
  {
    $filters = [];
    $attributes = [];
    $term_count = 0;

    if (!is_array($chosen_attributes) || empty($chosen_attributes)) {
      return [
        'filters' => $filters,
        'attributes' => $attributes,
        'term_count' => $term_count
      ];
    }

    foreach ($chosen_attributes as $taxonomy => $data) {
      // Only process product attributes (taxonomies starting with pa_)
      if (strpos($taxonomy, 'pa_') !== 0) {
        continue;
      }

      // Get attribute name without 'pa_' prefix
      $attribute = str_replace('pa_', '', $taxonomy);

      // Check if terms exist
      if (!isset($data['terms']) || !is_array($data['terms']) || empty($data['terms'])) {
        continue;
      }

      // Initialize attribute array if not exists
      if (!isset($filters[$attribute])) {
        $filters[$attribute] = [];
        $attributes[] = $attribute;
      }

      // Add terms to filters
      foreach ($data['terms'] as $term) {
        $filters[$attribute][] = $term;
        $term_count++;
      }
    }

    return [
      'filters' => $filters,
      'attributes' => $attributes,
      'term_count' => $term_count
    ];
  }

  public function robots($robots)
  {

    global $wp_query;

    // Get current URL for debugging
    $current_url = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://$_SERVER[HTTP_HOST]$_SERVER[REQUEST_URI]";

    // Get active filters from WooCommerce - this handles filter_* URL parameters
    $chosen_attributes = [];
    if (seolaxy_crawler_budget_control_is_woocommerce_active() && class_exists('WC_Query')) {
      $chosen_attributes = WC_Query::get_layered_nav_chosen_attributes();
    }

    // Transform chosen attributes to filter data format
    $filter_data = $this->transform_chosen_attributes_to_filter_data($chosen_attributes);

    // Check if we should process robots meta
    if (!$this->should_process_robots($filter_data['term_count'])) {
      return $robots;
    }

    // Get meta values for terms and attributes
    $term_meta_values = $this->get_term_meta_values($filter_data['filters']);
    $attributes_meta_values = $this->get_attribute_meta_values($filter_data['attributes']);

    // Process single attribute
    $single_result = $this->process_single_attribute($term_meta_values, $attributes_meta_values);
    if ($single_result) {
      if (isset($single_result['noindex'])) {
        $robots['noindex'] = true;
      }
      if ($single_result['remove_index']) {
        unset($robots['index']);
      }
      $this->logger->debug('robots() - Applied noindex for single attribute - URL: ' . $current_url . ', final robots: ' . json_encode($robots));
      return $robots;
    }

    // Process attribute combination
    $combo_result = $this->process_attribute_combination($attributes_meta_values);
    if ($combo_result) {
      if (isset($combo_result['index'])) {
        $robots['index'] = true;
      }
      if ($combo_result['remove_noindex']) {
        unset($robots['noindex']);
      }
      $this->logger->debug('robots() - Applied index for attribute combination - URL: ' . $current_url . ', final robots: ' . json_encode($robots));
      return $robots;
    }

    return $robots;
  }

  /**
   * Setup SEO plugin conflict resolution
   * 
   * Detects and handles conflicts with other SEO plugins like Rank Math
   * by ensuring our settings take priority.
   */
  public function setup_seo_conflicts_resolution()
  {
    // Check if Rank Math is active
    if ($this->is_rank_math_active()) {
      // Remove Rank Math's actions early
      add_action('wp_head', array($this, 'remove_rank_math_seo_output'), 1);

      // Add our output with higher priority (after removing Rank Math's)
      add_action('wp_head', array($this, 'output_robots_meta'), 5);
    }
    // Note: We don't need canonical handling for this plugin as it only handles robots
  }

  /**
   * Check if Rank Math SEO plugin is active
   * 
   * @return bool True if Rank Math is active
   */
  private function is_rank_math_active()
  {
    // By now (wp hook), all plugins should be loaded, so these checks should work
    return defined('RANK_MATH_VERSION') ||
      class_exists('RankMath') ||
      function_exists('rank_math') ||
      class_exists('\\RankMath\\Frontend\\Head');
  }

  /**
   * Remove Rank Math's SEO output from their custom hooks
   * 
   * This prevents Rank Math from outputting robots tags
   * so our plugin can take over with its own settings.
   */
  public function remove_rank_math_seo_output()
  {
    // Remove from rank_math/head action using object reference
    if (function_exists('rank_math') && isset(rank_math()->head)) {
      remove_action('rank_math/head', array(rank_math()->head, 'robots'), 10);
    }

    // Remove all actions from rank_math/head hook for robots priority
    remove_all_actions('rank_math/head', 10);

    // Prevent rank_math/head hook execution
    add_filter('rank_math/head', '__return_false', 1);
    add_action('rank_math/head', array($this, 'prevent_rank_math_output'), 1);
  }

  /**
   * Prevent Rank Math from outputting anything by stopping action execution
   */
  public function prevent_rank_math_output()
  {
    // Remove all subsequent actions on this hook
    remove_all_actions('rank_math/head');
  }

  /**
   * Output robots meta tag directly to wp_head
   * 
   * Used when wp_robots filter is not available (e.g., when Rank Math removes it)
   */
  public function output_robots_meta()
  {
    // Simulate the wp_robots filter behavior for our plugin
    $robots = array();
    $robots = $this->robots($robots);

    if (!empty($robots)) {
      $robots_content = implode(', ', array_keys(array_filter($robots)));
      if (!empty($robots_content)) {
        echo '<meta name="robots" content="' . esc_attr($robots_content) . '" />' . "\n";
      }
    }
  }

  /**
   * Register the JavaScript and stylesheets for the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function enqueue_public_resources()
  {
    /**
     * This function is provided for demonstration purposes only.
     *
     * An instance of this class should be passed to the run() function
     * defined in Seolaxy_Crawler_Budget_Control_Loader as all of the hooks are defined
     * in that particular class.
     *
     * The Seolaxy_Crawler_Budget_Control_Loader will then create the relationship
     * between the defined hooks and the functions defined in this
     * class.
     */
    wp_enqueue_style($this->id, SEOLAXY_CRAWLER_BUDGET_CONTROL_URL . 'public/css/seolaxy-crawler-budget-control-public.css', array(), $this->version, 'all');
    wp_enqueue_script($this->id, SEOLAXY_CRAWLER_BUDGET_CONTROL_URL . 'public/js/seolaxy-crawler-budget-control-public.js', array('jquery'), $this->version, false);
  }

  //PAGINATION

  function my_change_prev_next_text($args)
  {
    $posts_per_page = get_option('posts_per_page');

    $args['end_size'] = 0; // will display the number at the end
    $args['mid_size'] = $posts_per_page / 2; // will display the number before and after current number

    if (1 === 1) { //TODO - get from options page
      $args['prev_next'] = true;
    } else {
      $args['prev_next'] = false;
    }

    return $args;
  }

  function show_pagination($content)
  {
    $pagination_type = $this->seolaxy_crawler_budget_control_options['pagination'] ?? 'default';
    if ($pagination_type === 'ghost') {
      return $this->show_ghost_pagination($content);
    } elseif ($pagination_type === 'all') {
      return $this->custom_pagination($content);
    } else {
      return $content;
    }
  }

  function custom_pagination($content)
  {
    global $wp_query;

    $total_pages = $wp_query->max_num_pages;
    $current_page = max(1, get_query_var('paged'));
    $links_display = 10;
    $start = 1;
    $end = $total_pages;

    if ($current_page > 5) {
      $start = max(1, $current_page - 4);
    }
    if ($total_pages > $links_display) {
      $end = min($total_pages, $start + $links_display - 1);
    }

    $start_end_diff = $end - $start;

    if ($start_end_diff < ($links_display - 1) && ($end - $links_display + 1) > 0) {
      $start = $end - $links_display + 1;
    }

    $pagination_bg_color = $this->seolaxy_crawler_budget_control_options['pagination_bg_color'] ?? "";
    $pagination_font_color = $this->seolaxy_crawler_budget_control_options['pagination_font_color'] ?? "";

    $links = "
    <style>
      .woocommerce-pagination .page-numbers li a.page-numbers {
        background-color: $pagination_bg_color;
        color: $pagination_font_color;
        margin-right: 1px;
        margin-bottom: 1px;
      }
      .woocommerce-pagination .page-numbers li a.page-numbers:hover {
        background-color: $pagination_bg_color;
        opacity: 0.85;
      }
      
      .woocommerce-pagination .page-numbers li span.page-numbers.current, 
      .woocommerce-pagination .page-numbers li .page-numbers:not(.current) {
        color: $pagination_font_color;
        margin-right: 1px;
        margin-bottom: 1px;
      }
      
      .woocommerce-pagination .page-numbers li span.page-numbers.current {
        background-color: $pagination_bg_color;
        opacity: 0.85;
      }
    </style>
    ";

    $links .= '<ul class="page-numbers">';

    for ($i = $start; $i <= $end; $i++) {
      $links .= '<li>';
      if ($i == $current_page) {
        $links .= '<span class="page-numbers current">' . $i . '</span>';
      } else {
        $links .= '<a class="page-numbers" href="' . get_pagenum_link($i) . '">' . $i . '</a>';
      }
      $links .= "</li>";
    }
    $links .= '</ul>';

    return $links;
  }

  function show_ghost_pagination($content)
  {

    $content = $this->custom_pagination($content);

    global $wp_query;
    $last_page = $wp_query->max_num_pages;
    $paged = get_query_var('paged') ? get_query_var('paged') : 1;

    $from = intval($paged) * 10 + 1;
    $to = $from + 10 - 1;

    if ($from > $last_page) {
      return $content;
    }

    if ($to > $last_page) {
      $to = $last_page;
    }

    $content .= '<ul class="page-numbers">';
    $site_url = get_site_url();
    for ($page = $from; $page <= $to; $page++) {
      $paged_link = $this->build_link($page);

      $content .= '<li>';
      $content .= '<a class="page-numbers" href="' . $paged_link . '">' . $page . '</a>';
      $content .= '</li>';
    }
    $content .= '</ul>';
    return $content;
  }

  //TODO - END temp, remove

  function pagination_remove_first_page_from_url($link)
  {
    if (is_paged()) {
      // Check if permalinks are set to plain or rewritten
      $permalink_structure = get_option('permalink_structure');
      if (empty($permalink_structure)) {
        // Permalinks are set to "Plain"
        // Remove paged=1 query string

        // Parse the URL to retrieve the query parameters
        $query_params = wp_parse_url($link, PHP_URL_QUERY);
        parse_str($query_params, $query_args);

        // Check if the 'paged' parameter exists and has a value of 1
        if (isset($query_args['paged']) && $query_args['paged'] === '1') {
          // Remove the 'paged' parameter from the query arguments
          unset($query_args['paged']);

          // Rebuild the modified query parameters
          $modified_query_params = http_build_query($query_args);

          // Reconstruct the URL with the modified query parameters
          $link = strtok($link, '?') . '?' . $modified_query_params;
        }
      } else {
        // Permalinks are set to "Rewritten"
        // Remove page/1
        $pagination_base = $GLOBALS['wp_rewrite']->pagination_base;
        $link = str_replace($pagination_base . '/1/', '', $link);
      }
    }
    return $link;
  }


  function rel_next_prev()
  {
    global $paged;

    if (seolaxy_crawler_budget_control_is_woocommerce_active() && !is_shop()) { //TODO - Should this be included in shop index, like everywhere where we have a pagination?
      if (get_previous_posts_link()) { ?>
        <link rel="prev" href="<?php echo get_pagenum_link($paged - 1); ?>" />
      <?php
      }

      if (get_next_posts_link()) { ?>
        <link rel="next" href="<?php echo get_pagenum_link($paged + 1); ?>" />
<?php
      }
    }
  }

  function build_link($page)
  {
    $site_url = get_site_url();
    $pagination_base = $GLOBALS['wp_rewrite']->pagination_base;
    $permalink_structure = get_option('permalink_structure');

    if ($permalink_structure) {
      // Pretty permalinks are in use
      $new_url = trailingslashit($site_url) . $pagination_base . '/' . $page . '/';
    } else {
      // Plain permalinks
      $new_url = add_query_arg('paged', $page, $site_url);
    }

    $query_args = $_GET;
    unset($query_args['paged']);

    return add_query_arg($query_args, $new_url);
  }

  public function modify_category_title_with_filters($title)
  {
    if (is_product_category() && isset($this->seolaxy_crawler_budget_control_options['shop_category_show_attribute_filter_prefix']) && $this->seolaxy_crawler_budget_control_options['shop_category_show_attribute_filter_prefix']) {
      $prefix = $this->get_attribute_filter_prefix();
      if (!empty($prefix)) {
        return $prefix . ' ' . $title;
      }
    }
    return $title;
  }
  private function get_attribute_filter_prefix()
  {
    if (!is_product_category()) {
      return '';
    }

    $selected_terms = [];

    // Get the main query's tax query
    global $wp_query;
    $tax_queries = $wp_query->tax_query->queries;

    if (empty($tax_queries)) {
      return '';
    }

    foreach ($tax_queries as $query) {
      // Check if this is a product attribute query
      if (isset($query['taxonomy']) && strpos($query['taxonomy'], 'pa_') === 0) {
        // Get only the selected terms
        if (!empty($query['terms']) && is_array($query['terms'])) {
          $terms = get_terms([
            'taxonomy' => $query['taxonomy'],
            'slug' => $query['terms'],
            'hide_empty' => false
          ]);

          if (!is_wp_error($terms)) {
            foreach ($terms as $term) {
              $selected_terms[] = $term->name;
            }
          }
        }
      }
    }

    return implode(', ', array_unique($selected_terms));
  }
}
