<?php

if (!defined('ABSPATH')) {
  exit; // Exit if accessed directly
}

/**
 * The file that defines the core plugin class
 *
 * A class definition that includes attributes and functions used across both the
 * public-facing side of the site and the admin area.
 *
 * @link       https://www.seolaxy.com/
 * @since      1.0.0
 *
 * @package    Seolaxy_Crawler_Budget_Control
 * @subpackage Seolaxy_Crawler_Budget_Control/includes
 */

/**
 * The core plugin class.
 *
 * This is used to define internationalization, admin-specific hooks, and
 * public-facing site hooks.
 *
 * Also maintains the unique identifier of this plugin as well as the current
 * version of the plugin.
 *
 * @since      1.0.0
 * @package    Seolaxy_Crawler_Budget_Control
 * @subpackage Seolaxy_Crawler_Budget_Control/includes
 * @author     SEOLAXY
 */
class Seolaxy_Crawler_Budget_Control
{

  /**
   * The loader that's responsible for maintaining and registering all hooks that power
   * the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      Seolaxy_Crawler_Budget_Control_Loader    $loader    Maintains and registers all hooks for the plugin.
   */
  protected $loader;

  /**
   * The unique identifier of this plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $plugin_name    The string used to uniquely identify this plugin.
   */
  protected $plugin_name;

  /**
   * The current version of the plugin.
   *
   * @since    1.0.0
   * @access   protected
   * @var      string    $version    The current version of the plugin.
   */
  protected $version;

  protected $seolaxy_crawler_budget_control_options_values;

  /**
   * Define the core functionality of the plugin.
   *
   * Set the plugin name and the plugin version that can be used throughout the plugin.
   * Load the dependencies, define the locale, and set the hooks for the admin area and
   * the public-facing side of the site.
   *
   * @since    1.0.0
   */
  public function __construct()
  {
    $this->version = SEOLAXY_CRAWLER_BUDGET_CONTROL_VERSION;
    $this->plugin_name = 'seolaxy-crawler-budget-control';

    $this->seolaxy_crawler_budget_control_options_values = get_option('seolaxy_crawler_budget_control_options');

    $this->load_dependencies();
    $this->set_locale();
    $this->define_include_hooks();
    $this->define_admin_hooks();
    $this->define_public_hooks();
  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Include the following files that make up the plugin:
   *
   * - Seolaxy_Crawler_Budget_Control_Loader. Orchestrates the hooks of the plugin.
   * - Seolaxy_Crawler_Budget_Control_i18n. Defines internationalization functionality.
   * - Seolaxy_Crawler_Budget_Control_Admin. Defines all hooks for the admin area.
   * - Seolaxy_Crawler_Budget_Control_Public. Defines all hooks for the public side of the site.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function load_dependencies()
  {

    /**
     * Plugin Core Functions.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'includes/functions.php';

    /**
     * The class responsible for orchestrating the actions and filters of the
     * core plugin.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'includes/class-seolaxy-crawler-budget-control-loader.php';

    /**
     * The class responsible for defining internationalization functionality
     * of the plugin.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'includes/class-seolaxy-crawler-budget-control-i18n.php';

    /**
     * The class responsible for defining all actions that occur in both admin and public-facing areas.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'includes/class-seolaxy-crawler-budget-control-include.php';

    /**
     * The class responsible for defining all actions that occur in the admin area.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'admin/class-seolaxy-crawler-budget-control-admin.php';

    /**
     * The class responsible for defining all actions that occur in the public-facing
     * side of the site.
     */
    require_once SEOLAXY_CRAWLER_BUDGET_CONTROL_PATH . 'public/class-seolaxy-crawler-budget-control-public.php';

    $this->loader = new Seolaxy_Crawler_Budget_Control_Loader();
  }

  /**
   * Define the locale for this plugin for internationalization.
   *
   * Uses the Seolaxy_Crawler_Budget_Control_i18n class in order to set the domain and to register the hook
   * with WordPress.
   *
   * @since    1.0.0
   * @access   private
   */
  private function set_locale()
  {

    $plugin_i18n = new Seolaxy_Crawler_Budget_Control_i18n();

    $this->loader->add_action('plugins_loaded', $plugin_i18n, 'load_plugin_textdomain');
  }

  /**
   * Register all of the hooks related to both admin and public-facing areas functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_include_hooks()
  {

    $plugin_admin = new Seolaxy_Crawler_Budget_Control_Include($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('init', $plugin_admin, 'init_something');
  }

  /**
   * Register all of the hooks related to the admin area functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_admin_hooks()
  {

    $plugin_admin = new Seolaxy_Crawler_Budget_Control_Admin($this->get_plugin_name(), $this->get_version());

    $this->loader->add_action('admin_menu', $plugin_admin, 'add_admin_menu');
    $this->loader->add_action('admin_enqueue_scripts', $plugin_admin, 'enqueue_resources');

    /*Register Settings*/
    $this->loader->add_action('rest_api_init', $plugin_admin, 'register_settings');
    $this->loader->add_action('admin_init', $plugin_admin, 'register_settings');

    $this->loader->add_action('init', $this, 'add_custom_fields_to_all_attributes');

    $this->loader->add_action('rest_api_init', $this, 'register_rest_routes');
  }

  // ATTRIBUTES
  public function add_custom_fields_to_all_attributes()
  {
    // Only proceed if WooCommerce is active
    if (!seolaxy_crawler_budget_control_is_woocommerce_active()) {
      return;
    }

    add_action('woocommerce_after_add_attribute_fields', array($this, 'custom_attribute_add_field'), 10, 1);
    add_action('woocommerce_after_edit_attribute_fields', array($this, 'custom_attribute_edit_field'), 10, 1);

    add_action('woocommerce_attribute_added', array($this, 'save_custom_attribute_meta'), 10, 3);
    add_action('woocommerce_attribute_updated', array($this, 'save_custom_attribute_meta'), 10, 3);


    if (
      !isset($this->seolaxy_crawler_budget_control_options_values['attribute_terms_noindex']) ||
      $this->seolaxy_crawler_budget_control_options_values['attribute_terms_noindex']
    ) {
      // Get all WooCommerce attributes
      $attribute_taxonomies = wc_get_attribute_taxonomies();

      if (!empty($attribute_taxonomies)) {
        foreach ($attribute_taxonomies as $attr) {
          $taxonomy_name = wc_attribute_taxonomy_name($attr->attribute_name);

          // Hook into the add form for each attribute taxonomy
          add_action("{$taxonomy_name}_add_form_fields", array($this, 'add_custom_field_to_attribute_add_form'), 10, 1);
          // Hook into the edit form for each attribute taxonomy
          add_action("{$taxonomy_name}_edit_form_fields", array($this, 'add_custom_field_to_attribute_edit_form'), 10, 1);

          // Hook to save custom field value when a new term is created
          add_action("created_{$taxonomy_name}", array($this, 'save_custom_field_meta'), 10, 2);
          // Hook to save custom field value when a term is edited
          add_action("edited_{$taxonomy_name}", array($this, 'save_custom_field_meta'), 10, 2);
        }
      }
    }
  }
  //TERMS
  // For the add form
  function add_custom_field_to_attribute_add_form($taxonomy)
  {
?>
    <div class="form-field">
      <label for="seolaxy_noindex_term">
        <input type="checkbox" name="seolaxy_noindex_term" id="seolaxy_noindex_term" value="1" />
        <?php _e('Noindex', 'seolaxy-crawler-budget-control'); ?>
      </label>
      <p class="description"><?php _e('Set value of this filter (facet) to "noindex"', 'seolaxy-crawler-budget-control'); ?></p>
    </div>
  <?php
  }

  // For the edit form
  function add_custom_field_to_attribute_edit_form($term)
  {
    $seolaxy_noindex_term_value = get_term_meta($term->term_id, 'seolaxy_noindex_term', true);
  ?>
    <tr class="form-field">
      <th scope="row" valign="top"><label for="seolaxy_noindex_term"><?php _e('Noindex', 'seolaxy-crawler-budget-control'); ?></label></th>
      <td>
        <input type="checkbox" name="seolaxy_noindex_term" id="seolaxy_noindex_term" value="1" <?php checked($seolaxy_noindex_term_value, 1); ?> />
        <p class="description"><?php _e('Set value of this filter (facet) to "noindex"', 'seolaxy-crawler-budget-control'); ?></p>
      </td>
    </tr>
    <?php
  }

  // Save the custom field value
  function save_custom_field_meta($term_id)
  {
    // Check if the 'seolaxy_noindex_term' checkbox was posted.
    if (isset($_POST['seolaxy_noindex_term'])) {
      update_term_meta($term_id, 'seolaxy_noindex_term', sanitize_text_field($_POST['seolaxy_noindex_term']));
    } else {
      // If checkbox was not checked, update the meta value to '0' or false.
      update_term_meta($term_id, 'seolaxy_noindex_term', '0');
    }
  }


  //ATTRIBUTES

  function save_custom_attribute_meta($attribute_id, $data)
  {
    // Check if our custom checkbox was posted
    if (isset($_POST['seolaxy_noindex_attribute'])) {
      // Sanitize the value if necessary
      $value = sanitize_text_field($_POST['seolaxy_noindex_attribute']) ? 1 : 0;
      // Save custom meta for the attribute
      update_option('seolaxy_noindex_attribute_' . $attribute_id, $value);
    } else {
      update_option('seolaxy_noindex_attribute_' . $attribute_id, 0);
    }

    if (isset($_POST['seolaxy_filterkombi_attribute'])) {
      // Sanitize the value if necessary
      $value = sanitize_text_field($_POST['seolaxy_filterkombi_attribute']) ? 1 : 0;
      // Save custom meta for the attribute
      update_option('seolaxy_filterkombi_attribute_' . $attribute_id, $value);
    } else {
      update_option('seolaxy_filterkombi_attribute_' . $attribute_id, 0);
    }
  }

  function custom_attribute_add_field($taxonomy)
  {
    if (
      !isset($this->seolaxy_crawler_budget_control_options_values['attributes_noindex']) ||
      $this->seolaxy_crawler_budget_control_options_values['attributes_noindex']
    ) {
    ?>
      <div class="form-field">
        <label for="seolaxy_noindex_attribute">
          <input type="checkbox" name="seolaxy_noindex_attribute" id="seolaxy_noindex_attribute" />
          <?php _e('Noindex', 'seolaxy-crawler-budget-control'); ?>
        </label>
        <p class="description"><?php _e('Set all values of one filter (facet) to "noindex"', 'seolaxy-crawler-budget-control'); ?></p>
      </div>
    <?php
    }
    if (
      !isset($this->seolaxy_crawler_budget_control_options_values['attributes_filterkombi']) ||
      $this->seolaxy_crawler_budget_control_options_values['attributes_filterkombi']
    ) {
    ?>
      <div class="form-field">
        <label for="seolaxy_filterkombi_attribute">
          <input type="checkbox" name="seolaxy_filterkombi_attribute" id="seolaxy_filterkombi_attribute" />
          <?php _e('Allow double filter combinations', 'seolaxy-crawler-budget-control'); ?>
        </label>
        <p class="description"><?php _e('Activate "index" for a combination of two filter (facet) values if both values are set  to "index"', 'seolaxy-crawler-budget-control'); ?></p>
      </div>
    <?php
    }
  }

  function custom_attribute_edit_field()
  {
    $term_id = isset($_GET['edit']) ? absint($_GET['edit']) : 0;
    $seolaxy_noindex_attribute_value = get_option('seolaxy_noindex_attribute_' . $term_id);
    $seolaxy_filterkombi_attribute_value = $seolaxy_noindex_attribute_value ? 0 : get_option('seolaxy_filterkombi_attribute_' . $term_id);
    if (
      !isset($this->seolaxy_crawler_budget_control_options_values['attributes_noindex']) ||
      $this->seolaxy_crawler_budget_control_options_values['attributes_noindex']
    ) {
    ?>
      <tr class="form-field">
        <th scope="row"><label for="seolaxy_noindex_attribute"><?php _e('Noindex', 'seolaxy-crawler-budget-control'); ?></label></th>
        <td>
          <input type="checkbox" name="seolaxy_noindex_attribute" id="seolaxy_noindex_attribute" <?php checked($seolaxy_noindex_attribute_value, 1); ?> />
          <p class="description"><?php _e('Set all values of one filter (facet) to "noindex"', 'seolaxy-crawler-budget-control'); ?></p>
        </td>
      </tr>
    <?php
    }
    if (
      !isset($this->seolaxy_crawler_budget_control_options_values['attributes_filterkombi']) ||
      $this->seolaxy_crawler_budget_control_options_values['attributes_filterkombi']
    ) {
    ?>
      <tr class="form-field">
        <th scope="row"><label for="seolaxy_filterkombi_attribute"><?php _e('Allow double filter combinations', 'seolaxy-crawler-budget-control'); ?></label></th>
        <td>
          <input type="checkbox" name="seolaxy_filterkombi_attribute" id="seolaxy_filterkombi_attribute" <?php checked($seolaxy_filterkombi_attribute_value, 1);
                                                                                                          disabled($seolaxy_noindex_attribute_value, 1); ?> />
          <p class="description"><?php _e('Activate "index" for a combination of two filter (facet) values if both values are set  to "index"', 'seolaxy-crawler-budget-control'); ?></p>
        </td>
      </tr>
<?php
    }
  }

  /**
   * Register all of the hooks related to the public-facing functionality
   * of the plugin.
   *
   * @since    1.0.0
   * @access   private
   */
  private function define_public_hooks()
  {

    $plugin_public = new Seolaxy_Crawler_Budget_Control_Public($this->get_plugin_name(), $this->get_version());

    // Always register non-WooCommerce hooks
    $this->loader->add_action('wp_enqueue_scripts', $plugin_public, 'enqueue_public_resources');
    //$this->loader->add_action('wp_head', $plugin_public, 'set_schemas');

    $this->loader->add_action('wp_head', $plugin_public, 'rel_next_prev');
    $this->loader->add_filter('paginate_links', $plugin_public, 'pagination_remove_first_page_from_url', 10, true);
    $this->loader->add_filter('paginate_links_output', $plugin_public, 'show_pagination');

    // Register WooCommerce-specific hooks after all plugins are loaded
    $this->loader->add_action('plugins_loaded', $this, 'register_woocommerce_hooks');
  }

  /**
   * Register WooCommerce-specific hooks after all plugins are loaded
   *
   * @since    1.0.0
   */
  public function register_woocommerce_hooks()
  {
    // Now WooCommerce is loaded, safe to check and register hooks
    if (seolaxy_crawler_budget_control_is_woocommerce_active()) {
      $plugin_public = new Seolaxy_Crawler_Budget_Control_Public($this->get_plugin_name(), $this->get_version());

      // Register directly with WordPress since loader has already run
      add_filter('woocommerce_pagination_args', array($plugin_public, 'my_change_prev_next_text'), 99999999, true); // TODO - TEMP remove
      add_filter('woocommerce_page_title', array($plugin_public, 'modify_category_title_with_filters'), 20);
    }
  }

  /**
   * Run the loader to execute all of the hooks with WordPress.
   *
   * @since    1.0.0
   */
  public function run()
  {
    $this->loader->run();
  }

  /**
   * The name of the plugin used to uniquely identify it within the context of
   * WordPress and to define internationalization functionality.
   *
   * @since     1.0.0
   * @return    string    The name of the plugin.
   */
  public function get_plugin_name()
  {
    return $this->plugin_name;
  }

  /**
   * The reference to the class that orchestrates the hooks with the plugin.
   *
   * @since     1.0.0
   * @return    Seolaxy_Crawler_Budget_Control_Loader    Orchestrates the hooks of the plugin.
   */
  public function get_loader()
  {
    return $this->loader;
  }

  /**
   * Retrieve the version number of the plugin.
   *
   * @since     1.0.0
   * @return    string    The version number of the plugin.
   */
  public function get_version()
  {
    return $this->version;
  }

  public function register_rest_routes()
  {
    // Add license verification endpoint - updated pattern to handle URL-encoded keys
    register_rest_route($this->plugin_name . '/v1', '/verify-license/(?P<key>[^/]+)', [
      'methods' => 'GET',
      'callback' => [$this, 'verify_license'],
      'permission_callback' => function () {
        return current_user_can('manage_options');
      },
      'args' => [
        'key' => [
          'required' => true,
          'validate_callback' => function ($param) {
            return is_string($param);
          },
          'sanitize_callback' => function ($param) {
            // URL decode the license key for processing
            return urldecode($param);
          }
        ]
      ]
    ]);
  }

  public function verify_license($request)
  {
    $license_key = $request->get_param('key');
    $domain = parse_url(get_site_url(), PHP_URL_HOST);

    // URL encode the license key for the external API call
    $encoded_license_key = urlencode($license_key);

    $response = wp_remote_get(
      "https://api.bigburg.bosstech.info/plugin/verify/{$encoded_license_key}/{$domain}",
      array(
        'timeout' => 15,
        'headers' => array(
          'Accept' => 'application/json'
        )
      )
    );

    if (is_wp_error($response)) {
      return new WP_Error('license_verification_failed', $response->get_error_message());
    }

    $body = wp_remote_retrieve_body($response);

    if ($body === 'false') {
      return new WP_REST_Response([
        'active' => false,
        'error' => 'Invalid license key'
      ], 200);
    }

    $data = json_decode($body, true);
    if (json_last_error() === JSON_ERROR_NONE) {
      return $data;
    }

    return new WP_Error('invalid_response', 'Invalid response from license server');
  }
}
